function [s] = geeSector(c, top, bot, onePiece)
    psiT = c.halfT*%pi/180;
    east = c.r*cos(c.psi0 - psiT);
    west = -east;
    if onePiece then
        s = c.xyz;
        z = s(:,3);
        y = s(:,2);
        ndx = find((z <= top) & (z >= bot) & (y <= east) & (y >= west));
        s = s(ndx,:);
    else
        ndx = 181 + [-round(c.halfT):round(c.halfT)];
        s = c.xyz(ndx,:);
    end
        s = [s(1,:); s; s($,:); s(1,:)];
        s(1  ,1) = s(1  ,1) - (top-s(1  ,3))*tan(c.lambda);  s(1  ,3) = top;
        s($-1,1) = s($-1,1) - (top-s($-1,3))*tan(c.lambda);  s($-1,3) = top;
        s($  ,1) = s($  ,1) - (top-s($  ,3))*tan(c.lambda);  s($,3)   = top;
    c.s = s;
endfunction

function [c] = cone(lambda,w,e,f,halfT,bt)
    // lambda: latitude in radians
    // w: horizontal distance between bearings and cone apex
    // e: distance between bearings in mm
    // f: post height in mm
    tl = tan(lambda);
    cl = cos(lambda);
    sl = sin(lambda);
    
    function [xyz] = geeCircle(c)
        // Create the Gee circle that is the intersection of a cone
        // with a plane perpendicular to its axis through the bearings.
        // r: Circle radius
        // p: Horizontal distance between circle center and cone apex
        // q: Vertical distance between circle center and cone apex
        // f: Post height (the top of the post is at the cone apex)
        // lambda: Latitude of the observer
        // psi: Angles of rotation (clockwise as seen from the NCP) 
        // of the circle vertices (0 = West = Y+)
        // xyz: Coordinates of psi rotated around Y axis over lambda radians.
        //      xyz turns right in function of psi (as seen from the NCP)
        //      starting at the Y axis
        // g: Rotation around Y axis over lambda (right turn as seen from Y+)
        g = [ cos(c.lambda), 0, -sin(c.lambda);
              0            , 1, 0            ;
              sin(c.lambda), 0, cos(c.lambda)];
        n = length(c.psi);
        xyz = c.r*[zeros(c.psi),cos(c.psi),-sin(c.psi)];
        xyz = ones(c.psi)*[c.p, 0, c.q] + xyz*g';
    endfunction
    
    function [y] = rCost(x,w,f,cl,sl,tl)
        q = (w - f*tl)*cl*sl;
        r = (q+f)/cl + x^2;
        psi0 = asin((q + f)/(r*cl));
        y = e - r*cos(psi0);
    endfunction

    q = (w - f*tl)*cl*sl;
    r0 = 1.01*(q + f)/cl;
    // r >= (q+f)/cl must hold, let's reparameterize to force it:
    // r = (q+f)/cl + x^2
    // x = sqrt(r - (q+f)/cl)
    x0 = sqrt(r0 - (q+f)/cl);   // New parameter 
    try
        [dummy, x] = leastsq(list(rCost,w,f,cl,sl,tl), x0);
        r = (q+f)/cl + x^2;
        c.psi0 = asin((q + f)/(r*cl));
        c.phi = atan(e/w);
        c.p = (w - f*tl)*cl^2;
        c.q = q;
        c.r = r;
        c.f = f;
        c.w = w;
        c.e = e;
        c.lambda = lambda;
        c.psi = c.psi0 + [-180:180]'*%pi/180;
        c.lambda = lambda;
        c.xyz = geeCircle(c);
        c.halfT = halfT;
        c.bt = bt;
    catch
        c = [];
        return;
    end
endfunction

function [c] = coneNorth(lambda,d,u,e,f,halfT,bt,mt,mb,vns)
    // lambda: latitude in radians
    // d: center of mass in mm
    // u: horizontal distance from center of mass to bearings in mm
    // e: distance between bearings in mm
    // f: post height in mm

    function [y] = rstCost(x,w,u,d,mt,f,cl,sl,tl,halfT)
        st = x(2);
        w = u + (d + mt + st)/tl;
        w = u + d/tl;
        q = (w - f*tl)*cl*sl;
        r = (q+f)/cl + x(1)^2;
        psi0 = asin((q + f)/(r*cl));
        psiT = halfT*(%pi/180);
        stTarget = q - cl*r*sin(psi0 - psiT);
        y = [e - r*cos(psi0); stTarget - x(2)];
    endfunction

    // Find the top of the segment
    cl = cos(lambda);
    sl = sin(lambda);
    tl = tan(lambda);
    w = 4*d;    // Initial guess
    q = (w - f*tl)*cl*sl;
    r0 = 1.01*(q + f)/cl;
    // r >= (q+f)/cl must hold, let's reparameterize to force it:
    // r = (q+f)/cl + x^2
    // x = sqrt(r - (q+f)/cl)
    x0 = sqrt(r0 - (q+f)/cl);   // New parameter 
    st0 = 0;
    try
        [dummy, x] = leastsq(list(rstCost,w,u,d,mt,f,cl,sl,tl,halfT), [x0; st0]);
        st = x(2);
        w = u + (d + st + mt)/tl;
        c = cone(lambda, w, e, f, halfT, bt);
    catch
        c = [];
        return;
    end
    c.mt = mt;
    c.mb = mb;
    c.st = st;
    psiT = halfT*(%pi/180);
    c.sb = c.q - cl*c.r*sin(c.psi0 + psiT);
    c.d = d;
    c.u = u;
    c.cgBal = 1 - u/w;
    top = max(c.st, c.st+c.mt-c.bt);
    c.s = geeSector(c, top, c.sb, %f);
    c.vns = vns;
endfunction

function [c] = coneSouth(cNorth,e,v,halfT,vns)
    // lambda: latitude in radians
    // d: center of mass in mm
    // u: horizontal distance from center of mass to bearings in mm
    // e: distance between bearings in mm
    // f: post height in mm
    lambda = cNorth.lambda;
    w = cNorth.w - v;
    f = cNorth.f;
    c.halfT = halfT;
    c = cone(lambda, w, e, f, halfT, cNorth.bt);
    if c == [] then
        return
    end
    top = max(cNorth.st, cNorth.st+cNorth.mt-cNorth.bt);
    c.s = geeSector(c, top, cNorth.sb, ~vns);
    c.v = v;
    c.vns = vns;
endfunction

function [c] = vnsCone(c, top, foc)

    function [X, off] = sectorPlane(c)
        // Define the vertical plane tangent to the projection of the Gee disc
        // on the horizontal plane at the bearing, and the bearing projection.
        off = [c.w; c.e; 0];
        // X: Least squares regression matrix with columns that span the plane
        // NOTE: X must be orthogonal so (-mu,nu) are the plane coordinates
        // in other functions that use X
        X = [[sin(c.lambda)*cos(c.psi0);-sin(c.psi0);0], [0; 0; 1]];
        //X = [[c.e;-c.w;0], [0; 0; 1]];
        X(:, 1) = X(:, 1)/norm(X(:,1));
    endfunction

    function [sector, omRel] = vnsCurve(c, foc)
        // Projection of the Gee circle on a vertical plane tangent to
        // the circle at the bearing contact.  The projection is done using
        // intersections of the lines through the cone apex and the Gee
        // circle points with the sector plane.  This is for concave
        // sectors on the BOTTOM board.
        // sector: Circle projection  (columns are x,y,z)
        xyz = c.xyz;
        X = c.X;
        off = c.off;
        n = size(xyz, 1);
        w = c.w;
        f = c.f;
        e = c.e;
        lambda = c.lambda;
        sector = zeros(xyz);
        omRel = zeros(n,2);
        // v = xyz(i,:)'
        // off + X*[mu; nu] = v *sigma
        // [v, -X]*[sigma; mu; nu] = off
        // [sigma; mu; nu] = [v, -X]\off
        for i = 1:n do
            dpsi = c.psi(i) - c.psi0;
            B = polarRotation(lambda, dpsi);
            v = xyz(i,:)';
            if f == 0 then
                tmp = [v, -X]\off;
                sigma = tmp(1);
                mu = tmp(2);
                nu = tmp(3);
                sector(i, :) = (v*sigma)';
                omRel(i,1) = 1/sigma;
            else
                // Find the rotated-back sector (X1*[b;c] + off1)
                X1 = B'*X;
                off1 = B'*off;
                // Find the axis equations (axis*a + axisOff)
                //axis = [w; e; 0];
                //axisOff = [0;0;-f];
                // Intersect the rotated-back sector plane with the axis
                // a* + axisOff = off1 + X1*[b;c]
                tmp = [B*[w;e;0], -X]\([w;e;0] + B*[0;0;f]);
                sigma = tmp(1);
                mu = tmp(2);
                nu = tmp(3);
                // Forward rotate the result
                sector(i,:) = B*(sigma*[w;e;0] + [0;0;-f]);
                omRel(i,1) = norm([w;e;-f])/norm(sigma*[w;e;0] + [0;0;-f]);
            end
            // Calculate the angular velocity of the platform relative
            // to the nominal one (Earth's rotation)
            tmp = [0, -sin(lambda); 1, 0; 0, cos(lambda)];
            // Rotation matrix derivative
            dB = tmp*[-sin(dpsi), cos(dpsi); -cos(dpsi), -sin(dpsi)]*tmp';
            // Solve the derivative at psi
            d = [B*[w;e;0], -X]\(dB*([0;0;f] - [w;e;0]*sigma));
            // Rotate it back to the bearing
            ksiDot = dB*(sigma*[w;e;0] + [0;0;-f]) + B*d(1)*[w;e;0];
            h = B'*ksiDot;
            cosPhi = norm((h'/norm(h))*X);
            omRel(i,2) = cosPhi^2;
        end
        // Numerical profile derivative at the bearing
        /*for i = 1:n do
            dpsi = c.psi(i) - c.psi0;
            B = polarRotation(lambda, dpsi);

            X1 = B'*X;
            off1 = B'*off;
            // Find the axis equations (axis*a + axisOff)
            //axis = [w; e; 0];
            //axisOff = [0;0;-f];
            // Intersect the rotated-back sector plane with the axis
            // a* + axisOff = off1 + X1*[b;c]
            tmp = [B*[w;e;0], -X]\([w;e;0] + B*[0;0;f]);
            sigma = tmp(1);
            mu = tmp(2);
            nu = tmp(3);
            // Calculate the angular velocity of the platform relative
            // to the nominal one (Earth's rotation)
            tmp = [0, -sin(lambda); 1, 0; 0, cos(lambda)];
            // Rotation matrix derivative
            dB = tmp*[-sin(dpsi), cos(dpsi); -cos(dpsi), -sin(dpsi)]*tmp';
            // Solve the derivative at psi
            d1 = [B*[w;e;0], -X]\(dB*([0;0;f] - [w;e;0]*sigma));
            // Rotate it back to the bearing
            // Sector:  B*(sigma*[w;e;0] + [0;0;-f])
            d11 = dB*(sigma*[w;e;0] + [0;0;-f]) + B*d1(1)*[w;e;0];
            h1 = B'*d11;
            delta = 0.00001;
            dB1 = polarRotation(lambda, dpsi + delta);
            dB2 = polarRotation(lambda, dpsi - delta);
            dbNum = (dB1 - dB2)/(2*delta);

            if i == 1 then
                dp1 = sector(i+1,:) - sector(i,:);
                dp2 = sector(i,:) - sector(n,:);
                d = (dp1 + dp2)/(2*(c.psi(i+1)-c.psi(i)));
            elseif i == n then
                dp1 = sector(1,:) - sector(i,:);
                dp2 = sector(i,:) - sector(i-1,:);
                d = (dp1 + dp2)/(2*(c.psi(i)-c.psi(i-1)));
            else
                dp1 = sector(i+1,:) - sector(i,:);
                dp2 = sector(i,:) - sector(i-1,:);
                d = (dp1 + dp2)/(2*(c.psi(i)-c.psi(i-1)));
            end
            d = d';
            h = B'*d;
            cosPhi = norm((h'/norm(h))*X);
            omRel(i,2) = cosPhi;
            if i == 181 then
                pause
            end
            
        end*/
    endfunction

    function [s] = vnsSector(c, top)
        ndx = 181 + [-round(c.halfT):round(c.halfT)];
        s = c.sector(ndx,:);
        s = [s(1,:); s; s($,:); s(1,:)];
        s(1,3) = top;
        s($-1,3) = top;
        s($,3) = top;
    endfunction

    g = [ cos(lambda), 0, -sin(lambda);
          0          , 1, 0          ;
          sin(lambda), 0, cos(lambda)];
    [X, off] = sectorPlane(c);
    c.X = X;
    c.off = off;
    [sector, omRel] = vnsCurve(c, foc);
    c.foc = foc;
    c.sector = sector;
    c.s = vnsSector(c, top);
    c.omRel = omRel;
endfunction

function [b] = polarRotation(lambda, phi)
    // Find a rotation matrix for a rotation over phi degrees around the
    // polar axis (as seen from the NCP).  For increasing phi, this is in 
    // the same direction as the motion of the stars.
    // The polar axis is along [cos(lambda); 0; sin(lambda)]
    // 1) Rotate the polar axis onto X+ over lambda (left turn as seen from Y+)
    // 2) Rotate around X over an angle phi (positive = right turn looking from X+)
    // 3) Rotate X onto the polar axis over -lambda (right turn as seen from Y+)
    r1 = [ cos(lambda) , 0, sin(lambda);
           0           , 1,  0          ;
           -sin(lambda), 0, cos(lambda)];
    r2 = [ 1,  0       , 0       ;
           0,  cos(phi),  sin(phi);
           0, -sin(phi),  cos(phi)];
    b = r1'*r2*r1;
endfunction

function err = lineProjection(w, f ,e, xyz)
    // Axis: y = [0;0;-f] + [w;e;0]*theta
    // y + [0;0;-f] = [w;e;0]*theta
    // Projection: Y = X*w + e
    // e = Y - X*(X\Y)
    [m,n] = size(xyz);
    y = xyz';
    X = [w; e; 0];
    Y = y + [0;0;f]*ones(1,m);
    err = Y - X*(X\Y);
    err = min(sqrt(ones(1,3)*(err.*err)));
endfunction

function wcErr = classicError(psi2, psi0, e, sy1, sy2)
    // sy1: Sector profile of method 1
    // sy2: Sector profile of method 2
    // psi2: Gee angle amplitude in radians
    n = length(psi2);
    wcErr = zeros(n, 1);
    for i = 1:n do
        // Account for the angle of the platform 
        R = polarRotation(lambda, psi2(i) - psi0);
        eY = [0;1;0];   // Unit vector between bearings
        b = (R*eY)'*eY; // Inner product = cos of rotation angle
        // Let's assume look at the average absolute error of left and right
        errLeft = abs(sy1(i) - sy2(i));
        errRight = abs(sy1(n+1-i) - sy2(n+1-i));
        // The worst case error is roughly in the YZ plane since the
        // error effect rotates roughly around X.
        wcErr(i) = 0.5*(asin((errLeft*b)/(2*e)) + asin((errRight*b)/(2*e)));
    end
endfunction

function [cn,cs,butt] = vnsDesign(lambda,cog,en,es,u,v,f,tn,ts,bt,mt,mb,bw,nVns,sVns)
    cn = coneNorth(lambda,d=cog,u=u,e=en,f=f,halfT=tn,bt=bt,mt=mt,mb=mb,vns=nVns);
    if nVns then
        top = max(cn.st, cn.st+cn.mt-cn.bt);
        cn = vnsCone(cn, top, foc=%t);
    end
    if sVns then
        cs = coneSouth(cn,e=es,v=v,halfT=ts,vns=sVns);
        top = max(cn.st, cn.st+cn.mt-cn.bt);
        cs = vnsCone(cs, top, foc=%t);
        butt = cs.w - cn.f/tan(lambda) - 0.5*bw;    // End South board
    else
        cs = coneSouth(cn,e=es,v=v,halfT=ts,vns=sVns);
        butt = cs.w - 1.5*bw;
    end
endfunction
