// VNS and Gee equatorial platform design.
exec("vnsLib.sce");
exec("gfxLib.sce");

// Need some globals for the callbacks (perhaps, just too be sure).
// Otherwise, we pass them as if they were not globals.
global("cn","cs","vp","cmd");

// General parameters (cp = VNS Parameters).  Size units are in mm.
vp.dr = 315;            // Dobsonian base radius
vp.dfr = 15;            // Dobsonian feet radius
vp.dfd = 25;            // Dobsonian feet distance to Dob base edge
vp.simRes = 2;          // Simulation resolution (integer >= 1)
vp.simDel = 20;         // Simulation delay in ms per degree
vp.nSim = 1;            // Number of simulation cycles
vp.bw = 50;             // Half of board width
vp.butt = [];           // South end X coordinate of board
vp.geeGap = 5;          // Gap between Gee sector and support board
vp.nb2p = 550;          // Pivot to South board butt
vp.px = [];             // Pivot Z coordinate
vp.pz = [];             // Pivot Z coordinate
vp.version = "250225";  // Version
vp.paper = "Letter";    // Paper size (Letter or A4)

sd = 86164.0905;        // Sidereal day in seconds
om0 = 2*%pi/sd;         // Earth's angular velocity in radians/sec

// GUI and program flow parameters
cmd.doPrint = %f;
cmd.doSim = %f;
cmd.doGui = %t;
cmd.doHide = %f;
cmd.vnsTable = [];
cmd.vnsUi = [];
cmd.btnSim = [];
cmd.btnPrint = [];
cmd.btnHide = [];
cmd.ax = [];
cmd.fig3 = [];

// Configuration parameters (size in mm)
//
// - lambda: latitude in radians
// - cog: Dob center of mass to Dob base
// - en: separation between North bearings
// - es: separation between South bearings
// - u: North bearings to center of mass
// - v: North bearings to South bearings
// - f: Pivot (cone apex) height
// - tn: half the travel time in degrees
// - ts: the latter, times a factor to widen the South sector
// - bt: board thickness
// - mt: margin from sector top to Dob base
// - mb: margin from sector bottom to base top
// - bw: half the board width
// - nVns: %t for VNS, %f for Gee (North)
// - sVns: %t for VNS, %f for Gee (South)

if isfile("vns.dat") then
    // Only accept data files of the current version
    currentVersion = vp.version;
    load("vns.dat","cn","cs","vp");
    versionOk = isfield(vp, "version");
    if versionOk then
        versionOk = (vp.version == currentVersion);
    end
    if ~versionOk then
        messagebox("Please delete vnc.dat, it has a wrong version");
        return;
    end
    [cn,cs,butt] = vnsDesign(lambda=cn.lambda,cog=cn.d,en=cn.e, ...
        es=cs.e,u=cn.u,v=cs.v,f=cn.f,tn=cn.halfT,ts=cs.halfT, ...
        bt=cn.bt,mt=cn.mt,mb=cn.mb,bw=50, ...
        nTyp=cn.typ,sTyp=cs.typ);
else
    [cn,cs,vp.butt] = vnsDesign(lambda=34.416667*%pi/180,cog=324,en=225, ...
        es=70,u=175,v=425,f=75,tn=12,ts=1.2*12,bt=18,mt=30,mb=15,bw=50, ...
        nTyp="VNS",sTyp="Gee");
end

[cmd.fig3,cmd.ax,vp] = vnsFig(cn,cs,vp,cmd.fig3);

function [cn,cs,vp] = ui2data(cn,cs,vp,cmd)
    [mTbl, nTbl] = size(cmd.vnsTable);
    for i = 1:mTbl do
        cmdStr = cmd.vnsTable(i,3);
        str = cmd.vnsUi(i,2).string;
        execstr(cmdStr);
    end
    save("vns.dat","cn","cs","vp");
endfunction

function [cn,cs,vp,cmd] = data2ui(cn,cs,vp,cmd)
    cmd.vnsTable = [ ...
        "North type (VNS/Gee)", cn.typ,"cn.typ=str;"; ...
        "South type (VNS/Gee/Pivot)", cs.typ,"cs.typ=str;"; ...
        "Latitude", string(cn.lambda*180/%pi),"cn.lambda=strtod(str)*%pi/180"; ...
        "Center of mass", string(cn.d),"cn.d=strtod(str)"; ...
        "Cone apex to bearing plane", string(cn.f),"cn.f=strtod(str);cs.f=cn.f"; ...
        "North bearings to COM", string(cn.u),"cn.u=strtod(str)"; ...
        "North to South bearings", string(cs.v),"cs.v=strtod(str)"; ...
        "Travel time (North, min)", string(cn.halfT*2*4),"cn.halfT=round(strtod(str)/(2*4))"; ...
        "Travel time (South, min)", string(cs.halfT*2*4),"cs.halfT=round(strtod(str)/(2*4))"; ...
        "Top sector margin", string(cn.mt),"cn.mt=strtod(str)"; ...
        "Bottom sector margin", string(cn.mb),"cn.mb=strtod(str)"; ...
        "Bearing separation (North)", string(2*cn.e),"cn.e=strtod(str)/2"; ...
        "Bearing separation (South)", string(2*cs.e),"cs.e=strtod(str)/2"; ...
        "Board thickness", string(cn.bt),"cn.bt=strtod(str);cs.bt=strtod(str)"; ...
        "Board width", string(2*vp.bw),"vp.bw=strtod(str)/2"; ...
        "Dob base radius", string(vp.dr),"vp.dr=strtod(str)"; ...
        "Dob feet radius", string(vp.dfr),"vp.dfr=strtod(str)"; ...
        "Dob feet to base edge", string(vp.dfd),"vp.dfd=strtod(str)"; ...
        "Gee sector to backing", string(vp.geeGap),"vp.geeGap=strtod(str)"; ...
        "North bearings to pivot", string(vp.nb2p),"vp.nb2p=strtod(str)"; ...
        "Sim resolution", string(vp.simRes),"vp.simRes=round(strtod(str))"; ...
        "Sim sleep delay", string(vp.simDel),"vp.simDel=round(strtod(str))"; ...
        "Sim cycles", string(vp.nSim),"vp.nSim=strtod(str)"; ...
        "Print paper (Letter/A4)", vp.paper,"vp.paper=str"];
    cmd.fig3.immediate_drawing = "off";
    [mTbl,nTbl] = size(cmd.vnsTable);
    if cmd.vnsUi == [] then
        for i = 1:mTbl do
            k = mTbl+1-i;
            cmd.vnsUi(i, 1) = uicontrol(cmd.fig3, 'style', 'text', 'string', ...
                cmd.vnsTable(i, 1), 'position', [10, 30+k*20, 150, 20]);
            cmd.vnsUi(i, 2) = uicontrol(cmd.fig3, 'style', 'edit', 'string', ...
                cmd.vnsTable(i, 2),'position', [160, 30+k*20, 80, 20]);
            cmd.vnsUi(i,2).tag = cmd.vnsTable(i,1);
        end
    else
        for i = 1:mTbl do
            h = findobj("tag", cmd.vnsTable(i,1));
            h.string = cmd.vnsTable(i, 2);
        end
    end
    cmd.fig3.immediate_drawing = "on";
    drawnow();
endfunction

function [cn,cs,cmd,vp] = uiApply(cn,cs,vp,cmd)
    [cn,cs,vp] = ui2data(cn,cs,vp,cmd);
    [cn,cs,vp.butt] = vnsDesign(lambda=cn.lambda,cog=cn.d,en=cn.e, ...
        es=cs.e,u=cn.u,v=cs.v,f=cn.f,tn=cn.halfT,ts=cs.halfT, ...
        bt=cn.bt,mt=cn.mt,mb=cn.mb,bw=vp.bw, ...
        nTyp=cn.typ,sTyp=cs.typ);
    [cmd.fig3, cmd.ax, vp] = vnsFig(cn,cs,vp,cmd.fig3);
    db = zeros(2,3);
    topBoard = findobj("tag", "topBoard");
    bottomBoard = findobj("tag", "bottomBoard");
    dobBase = findobj("tag", "dobBase");
    northComMarkers = findobj("tag", "northComMarkers");
    nbm = findobj("tag", "northBearingMarkers");
    sbm = findobj("tag", "southBearingMarkers");
    pivot1 = findobj("tag", "pivot");
    xMin = min(min(topBoard.data.x), min(nbm.data(:,1)), min(sbm.data(:,1)), min(pivot1.data(:,1)));
    xMax = max(max(dobBase.data.x),max(topBoard.data.x));
    yMin = min(min(topBoard.data.y), min(dobBase.data.y));
    yMax = - yMin;
    zMin = min(bottomBoard.data.z);
    zMax = max(northComMarkers.data(:,3));
    cmd.ax.data_bounds = [xMin-10, yMin-10, zMin-10; xMax+10, yMax+10, zMax+10];
    delta = max(cn.s(:,3)) - min(topBoard.data.z);
    if delta > 0.001 then
        printf("WARNING: The segment sticks %6.2f mm into the top board - " + ...
            "increase the top sector margin.\n", delta);
    end
endfunction

function [cmd] = uiHide(cmd)
    cmd.fig3.immediate_drawing = "off"; // Does not seem to have any effect
    [mTbl, nTbl] = size(cmd.vnsTable);
    cmd.doHide = ~cmd.doHide;
    uiOnOff = ["on";"off"];
    for i = 1:mTbl do
        cmd.vnsUi(i, 1).visible = uiOnOff(cmd.doHide+1);
        cmd.vnsUi(i, 2).visible = uiOnOff(cmd.doHide+1);
    end
    cmd.btnApply.visible = uiOnOff(cmd.doHide+1);
    cmd.btnSim.visible = uiOnOff(cmd.doHide+1);
    cmd.btnPrint.visible = uiOnOff(cmd.doHide+1);
    cmd.fig3.immediate_drawing = "on";
    drawnow();
endfunction

// GUI
function [cmd] = uiShow(cn,cs,vp,cmd)
    cmd.fig3.immediate_drawing = "off";
    cmd.btnSim = uicontrol(cmd.fig3, 'style', 'pushbutton', 'position', [110 10 100 30], ...
                'string', "Simulate", 'callback_type', 2, ...
                'callback', "global(""cn"",""cs"",""vp"",""cmd"");" + ...
                "cmd.doSim = ~cmd.doSim;cmd=vnsSim(cn,cs,vp,cmd)");
    cmd.btnApply = uicontrol(cmd.fig3, 'style', 'pushbutton', 'position', [210 10 100 30], ...
                'string', "Apply/Save", ...
                'callback', "global(""cn"",""cs"",""vp"",""cmd"");" + ...
                "[cn,cs,cmd,vp]=uiApply(cn,cs,vp,cmd);cmd=uiShow(cn,cs,vp,cmd)");
    cmd.btnPrint = uicontrol(cmd.fig3, 'style', 'pushbutton', 'position', [310 10 100 30], ...
                'string', "Print", ...
                'callback', "global(""cn"",""cs"",""vp"",""cmd"");" + ...
                "cmd=uiHide(cmd);vnsPrint(cn,cs,vp,cmd.fig3)");
    cmd.btnHide = uicontrol(cmd.fig3, 'style', 'pushbutton', 'position', [10 10 100 30], ...
                'string', "Hide", 'callback', "global(""cmd"");cmd=uiHide(cmd)");
    [cn,cs,vp,cmd] = data2ui(cn,cs,vp,cmd);
endfunction

if cmd.doGui then
    cmd = uiShow(cn,cs,vp,cmd);
end

function [cmd] = vnsSim(cn,cs,vp,cmd)
    if cmd.doSim then
        dx = vp.butt;
        degStep = %pi/(vp.simRes*180);
        i = -vp.simRes*cn.halfT;
        turn = 1;
        dobBase = findobj("tag", "dobBase");
        northSectorEast = findobj("tag", "northSectorEast");
        northSectorWest = findobj("tag", "northSectorWest");
        topBoard = findobj("tag", "topBoard");
        dobCentral = findobj("tag", "dobCentral");
        dobFoot1 = findobj("tag", "dobFoot1");
        dobFoot2 = findobj("tag", "dobFoot2");
        dobFoot3 = findobj("tag", "dobFoot3");
        dobFoot4 = findobj("tag", "dobFoot4");
        dobFoot5 = findobj("tag", "dobFoot5");
        dobFoot6 = findobj("tag", "dobFoot6");
        northComLine = findobj("tag", "northComLine");
        northComMarkers = findobj("tag", "northComMarkers");
        southSectorEast = findobj("tag", "southSectorEast");
        southSectorWest = findobj("tag", "southSectorWest");
        iSim = 0;
        nCycle = 4*vp.simRes*cn.halfT;
        while cmd.doSim do
            b = polarRotation(cn.lambda, turn*degStep);
            rotateChild(dobBase, b, dx);
            rotateChild(northSectorEast, b, dx);
            rotateChild(northSectorWest, b, dx);
            rotateChild(topBoard, b, dx);
            rotateChild(dobCentral, b, dx);
            rotateChild(dobFoot1, b, dx);
            rotateChild(dobFoot2, b, dx);
            rotateChild(dobFoot3, b, dx);
            rotateChild(dobFoot4, b, dx);
            rotateChild(dobFoot5, b, dx);
            rotateChild(dobFoot6, b, dx);
            rotateWire(northComLine, b, dx);
            rotateWire(northComMarkers, b, dx);
            rotateChild(southSectorEast, b, dx);
            rotateChild(southSectorWest, b, dx);
            sleep(vp.simDel);
            i = i + 1;
            iSim = iSim + 1;
            cmd.doSim = (iSim < vp.nSim*nCycle);
            if abs(modulo(i, 2*vp.simRes*cn.halfT)) < 1e-10 then
                turn = -turn;
            end
            if vp.simRes == 1 then
                // Check if the profile touches the bearing
                data = [northSectorWest.data.x + dx, ...
                        northSectorWest.data.y, ...
                        northSectorWest.data.z];
                errNorth = lineProjection(cn.w,cn.f,cn.e,data);
                data = [southSectorWest.data.x + dx, ...
                        southSectorWest.data.y, ...
                        southSectorWest.data.z];
                errSouth = lineProjection(cs.w,cs.f,cs.e,data);
                printf("North error = %g, South error = %g\n", errNorth, errSouth);
            end
        end
    end
endfunction

if cmd.doSim then
    vnsSim(cn,cs,vp,cmd);
end

if cmd.doPrint then
    vnsPrint(cn,cs,vp,cmd.fig3);
end

return
