function [x,y,z] = topBoard(north, east, south, west, bt, top, bw)
    y = [east,west,west,-bw,-bw,bw,bw,east,east];
    x = [north,north,north-2*bw,north-2*bw,south,south,north-2*bw,north-2*bw,north];
    z = bt*ones(y);
    x = [x(1:$-1);x(2:$);x(2:$);x(1:$-1);x(1:$-1)];
    y = [y(1:$-1);y(2:$);y(2:$);y(1:$-1);y(1:$-1)];
    z = [bt;bt;0;0;bt]*ones(x(1,:)) + top;
endfunction

function [x, y, z] = cylinder(ctr, r, bt, n)
    t = [0:n]/n;
    y = r*cos(2*%pi*t);
    x = r*sin(2*%pi*t);
    z = zeros(1,n);
    x = [x(1:$-1);x(2:$);x(2:$);x(1:$-1);x(1:$-1)] + ctr(1);
    y = [y(1:$-1);y(2:$);y(2:$);y(1:$-1);y(1:$-1)] + ctr(2);
    z = [z;z;z+bt;z+bt;z] + ctr(3);
endfunction

function [x,y,z] = dobBase(r, bt, w, cgBal, high)
    n = 90;
    ctr = [cgBal*w; 0; high+bt];
    [x, y, z] = cylinder(ctr, r, bt, n);
endfunction

function [x1,y1,z1,x2,y2,z2,x3,y3,z3] = dobFeet(r,ctr,w,bt,cgBal,high)
    ctr(3) = high;
    [x1,y1,z1] = cylinder(ctr, r, bt, 20);
    phi = 2*%pi/3;
    c = cos(phi);
    s = sin(phi);
    R = [c, s, 0; -s, c, 0; 0,0,1];
    ctr(1) = ctr(1) - cgBal*w;
    ctr = R*ctr;
    ctr(1) = ctr(1) + cgBal*w;
    [x2,y2,z2] = cylinder(ctr, r, bt, 20);
    ctr(1) = ctr(1) - cgBal*w;
    ctr = R*ctr;
    ctr(1) = ctr(1) + cgBal*w;
    [x3,y3,z3] = cylinder(ctr, r, bt, 20);
endfunction

function rotateChild(child, b, dx, dz)
    data = child.data;
    [m, n] = size(data.x);
    x = data.x + dx;
    y = data.y;
    z = data.z + dz;
    newData = b*[x(:), y(:), z(:)]';
    x = matrix(newData(1,:), m, n);
    y = matrix(newData(2,:), m, n);
    z = matrix(newData(3,:), m, n);
    data.x = x - dx;
    data.y = y;
    data.z = z - dz;
    child.data = data;
endfunction

function rotateWire(child, b, dx, dz)
    data = child.data;
    data(:,1) = data(:,1) + dx;
    data(:,3) = data(:,3) + dz;
    data = data*b';
    data(:,1) = data(:,1) - dx;
    data(:,3) = data(:,3) - dz;
    child.data = data;
endfunction

function h = myPlot3d(tag,x,y,z)
    // x,y,z are matrices with line data in the columns
    h = findobj("tag", tag);
    if h == [] then
        h = plot3d(x, y, z);
        h.tag = tag;
    else
        data = h.data;
        data.x = x;
        data.y = y;
        data.z = z;
        h.data = data;
    end
endfunction

function h = myScatter3d(tag,x,y,z)
    // x,y,z are row vectors
    h = findobj("tag", tag);
    if h == [] then
        h = scatter3d(x, y, z, "fill");
        h.tag = tag;
    else
        h.data = [x, y, z];
    end
endfunction

function h = myParam3d(tag,x,y,z)
    // x,y,z are row vectors
    h = findobj("tag", tag);
    if h == [] then
        h = param3d(x, y, z, "fill");
        h.tag = tag;
    else
        h.data = [x, y, z];
    end
endfunction

function [fig3, ax, vp] = vnsFig(cn,cs,vp,fig3)
    totalT = 2*cn.halfT/15;
    // The sector top is at the bottom of the EQ top board
    // The Dob base is at the top of the EQ top board
    topBot = cn.sb - cn.mb;
    botTop = cn.st + cn.mt - cn.bt;
    east = max(cn.s(:,2));    // Board East
    west = -east;             // Boar West
    if cs.typ == "Pivot" then
        south = vp.butt;
    else
        south = cn.w - cn.u - vp.dr - 0.5*vp.bw;
    end
    north = cn.w + vp.bw;
    
    if cn.typ == "VNS" then
        // Tracking error over 240 seconds (1 degree) integrated over all degrees
        ndx = 181 + [-cn.halfT:cn.halfT];
        n = length(ndx);
        omr = cn.omRel(ndx,:);
        /*scf(1); clf(); plot([-cn.halfT:cn.halfT],omr, 'o-');
        legend("Distance","Angle");
        title("Relative slowdown");*/
        omr = omr(:,1); //.*omr(:,2);
        omr = (omr - 1);        // Deviation from nominal
        om = omr*om0*180/%pi;   // Degrees/s
        omInt = zeros(n,1);
        dt = 240;
        omInt(1) = om(1)*dt;
        for j = 2:n do
            // Each element is 1 degree is 4 minutes
            omInt(j) = omInt(j-1) + om(j)*dt;
        end
        printf("Tracking error = %5.2g degrees RMS, = %5.2g degrees (minmax)\n", ...
            stdev(omInt), max(omInt)-min(omInt));
        printf("Tracking error due to 5 mm sector thickness = %5.2g degrees (minmax)\n", ...
            (5/norm([cn.w;cn.e]))*om0*(cn.halfT*4*60)*180/%pi);
    end
    
    // Plot the EQ platform
    doCreate = %f;
    if (fig3 == []) then
        doCreate = %t;
    elseif ~is_handle_valid(fig3) then
        doCreate = %t;
    end
    if doCreate then
        fig3 = scf(2); clf();
        fig3.figure_size = [900,700];
    end
    fig3.immediate_drawing = "off";
    
    // South sectors
    x = cs.s(:,1);
    y = cs.s(:,2);
    z = cs.s(:,3);
    h = myPlot3d("southSectorWest", x, y, z);
    h.color_mode = 4;
    h.hiddencolor = 4;
    if cs.typ == "Pivot" then h.visible = "off"; else h.visible = "on"; end;
    if doCreate then
        ax = fig3.children(1);
        ax.tag = "axes";
    else
        ax = findobj("tag", "axes");
    end
    h = myPlot3d("southSectorEast", x, -y, z);
    h.visible = "off";
    if cs.typ == "VNS" then
        h.visible = "on";
    end
    if cs.typ == "Pivot" then h.visible = "off"; else h.visible = "on"; end;
    h.color_mode = 4;
    h.hiddencolor = 4;

    // Gee cone apex to bearings (North)
    x = [cn.w;0;cn.w];
    y = [-cn.e;0;cn.e];
    z = [-cn.f;0;-cn.f];
    h = myScatter3d("northBearingMarkers", x, y, z);
    h = myParam3d("northBearingLines", x, y, z);
    
    // Gee cone apex to bearings (South)
    x = [cs.w;0;cs.w];
    y = [-cs.e;0;cs.e];
    z = [-cs.f;0;-cs.f];
    h = myScatter3d("southBearingMarkers", x, y, z);
    if cs.typ == "Pivot" then h.visible = "off"; else h.visible = "on"; end;
    h = myParam3d("southBearingLines", x, y, z);
    if cs.typ == "Pivot" then h.visible = "off"; else h.visible = "on"; end;

    // Gee disc center to origin
    if cs.typ == "Pivot" then
        x = [0;cn.p];
        y = [0;0];
        z = [0;cn.q];
    else
        x = [0;cs.p;cn.p];
        y = [0;0;0];
        z = [0;cs.q;cn.q];
    end;
    h = myScatter3d("geeCenterMarkers", x, y, z);
    h = myParam3d("geeCenterLine", x, y, z);
    
    // Center of mass
    x = [cn.cgBal*cn.w;cn.cgBal*cn.w];
    y = [0;0];
    z = [cn.st+cn.mt;;cn.st+cn.mt+cn.d];
    h = myParam3d("northComMarkers", x, y, z);
    h.foreground = 5;
    h = myScatter3d("northComLine", x, y, z);
    h.mark_background = 5;
    
    // Dob feet
    ctr = [cn.cgBal*cn.w + vp.dr - vp.dfd; 0; 0];
    [x1,y1,z1,x2,y2,z2,x3,y3,z3] = dobFeet(vp.dfr,ctr,cn.w,cn.bt,cn.cgBal,botTop);
    h = myPlot3d("dobFoot1",x1,y1,z1);
    h = myPlot3d("dobFoot2",x2,y2,z2);
    h = myPlot3d("dobFoot3",x3,y3,z3);
    // Rotate them over %pi/3
    c = cos(%pi/3);
    s = sin(%pi/3);
    R = [c, s, 0; -s, c, 0; 0,0,1];
    ctr(1) = ctr(1) - cn.cgBal*cn.w;
    ctr = R*ctr;
    ctr(1) = ctr(1) + cn.cgBal*cn.w;
    [x1,y1,z1,x2,y2,z2,x3,y3,z3] = dobFeet(vp.dfr,ctr,cn.w,cn.bt,cn.cgBal,botTop);
    h = myPlot3d("dobFoot4",x1,y1,z1);
    h = myPlot3d("dobFoot5",x2,y2,z2);
    h = myPlot3d("dobFoot6",x3,y3,z3);
    // Central hole
    [x,y,z] = cylinder([cn.cgBal*cn.w;0;botTop],15,cn.bt,20);
    h = myPlot3d("dobCentral",x,y,z);
    
    // Top board
    [x,y,z] = topBoard(north, east, south, west, cn.bt, botTop, vp.bw);
    h = myPlot3d("topBoard",x,y,z)
    zBotTop = min(z);
    vp.butt = min(x);
    
    // Bottom board
    h = -cn.bt - topBot;
    z = z - (cn.st - cn.sb + cn.mt + cn.mb);
    h = myPlot3d("bottomBoard",x,y,z);
    zTopBot = max(z);
    tbGap = zBotTop - zTopBot;
    
    // Pivot
    vp.px = cn.w - vp.nb2p;
    vp.pz = vp.px*tan(cn.lambda);
    x = [vp.px;vp.px];
    y = [0;0];
    z = [topBot;vp.pz];
    h = myParam3d("pivot", x, y, z);
    h.thickness=3;
    h.foreground=9;
    if cs.typ == "Pivot" then
        h.visible = "on";
    else
        h.visible = "off";
    end
    h = myScatter3d("pivotMarkers", x, y, z)
    h.mark_background = 1;
    if cs.typ == "Pivot" then
        h.visible = "on";
    else
        h.visible = "off";
    end

    // North sectors
    h = myPlot3d("northSectorWest",cn.s(:,1),cn.s(:,2),cn.s(:,3));
    h.color_mode = 4;
    h.hiddencolor = 4;
    h = myPlot3d("northSectorEast",cn.s(:,1),-cn.s(:,2),cn.s(:,3));
    h.color_mode = 4;
    h.hiddencolor = 4;

    // Dob base
    [x,y,z] = dobBase(vp.dr, cn.bt, cn.w, cn.cgBal,botTop);
    h = myPlot3d("dobBase",x,y,z);
    
    // Gee backings (North)
    yWest = max(cn.s(:,2));
    yEast = min(cn.s(:,2));
    zTop = -cn.f - sin(cs.lambda)*vp.geeGap;
    zBot = topBot;
    xTop = cn.w - cos(cn.lambda)*vp.geeGap;
    xBot = xTop + tan(cn.lambda)*(zTop-zBot);
    x = [xBot;xBot;xTop;xTop;xBot];
    y = [yWest;yEast;yEast;yWest;yWest];
    z = [zBot;zBot;zTop;zTop;zBot];
    h = myParam3d("geeBackingNorthWest", x, y, z);
    tmp = "off";
    if cn.typ == "Gee" then
        tmp = "on";
    end
    h.visible = tmp;
    h = myParam3d("geeBackingNorthEast", x, -y, z);
    h.visible = tmp;

    // Gee backings (South)
    yWest = max(cs.s(:,2));
    yEast = min(cs.s(:,2));
    zTop = -cs.f - sin(cs.lambda)*vp.geeGap;
    zBot = topBot;
    xTop = cs.w - cos(cs.lambda)*vp.geeGap;
    xBot = xTop + tan(cs.lambda)*(zTop-zBot);
    x = [xBot;xBot;xTop;xTop;xBot];
    y = [yWest;yEast;yEast;yWest;yWest];
    z = [zBot;zBot;zTop;zTop;zBot];
    h = myParam3d("geeBackingSouth", x, y, z);
    h.visible = "off";
    if cs.typ == "Pivot" then 
        h.visible = "off"; 
    else 
        if cs.typ == "Gee" then
            h.visible = "on";
        end
    end;

    // Align the South butt with 0 to make construction easier
    h = findobj("tag", "bottomBoard");
    vp.dxButt = -vp.butt;
    vp.dzButt = -max(h.data.z);
    shiftXZ(ax, vp.dxButt, vp.dzButt);
    if doCreate then
        ax.data_bounds(:,1) = ax.data_bounds(:,1) + vp.dxButt;
        ax.data_bounds(:,3) = ax.data_bounds(:,3) + vp.dzButt;
        ax.rotation_angles = [65,45];
        ax.margins=[0.25,0.05,0.05,0.05];
    end

    ax.isoview = "on";
    ax.cube_scaling = "off";
    fig3.immediate_drawing = "on";
    drawnow();
    
    printf("Board separation = %6.2f mm\n", zBotTop - zTopBot);
    h = findobj("tag", "northBearingLines");
    printf("Cone apex = (%6.2f, %6.2f, %6.2f) mm\n", h.data(2,1), h.data(2,2), h.data(2,3));
    printf("Bearing height above the top of the bottom board = %6.2f mm\n", h.data(1,3));
    if cs.typ == "Pivot" then
        h = findobj("tag", "pivot");
        printf("Pivot height above the top of the bottom board = %6.2f mm\n", h.data(2,3));
    end

endfunction

function shiftXZ(ax, dx, dz)
    // Shift the X coordinates such that the South butt is at zero
    n = length(ax.children);
    ax = ax;
    for i = 1:n do
        h = ax.children(i);
        if typeof(h.data) == "3d" then
            h.data.x = h.data.x + dx;
            h.data.z = h.data.z + dz;
        else
            h.data(:,1) = h.data(:,1) + dx;
            h.data(:,3) = h.data(:,3) + dz;
        end
    end
endfunction

function [] = vnsPrint(cn, cs, vp, fig3)
    
    lambda = cn.lambda;

    // Letter size paper width/height, landscape/portrait
    if vp.paper == "Letter" then
        ltrw = 8.5*25.4;
        ltrh = 11*25.4;
    elseif vp.paper == "A4" then
        ltrw = 8.3*25.4;
        ltrh = 11.7*25.4;
    else
        messagebox("Invalid paper size");
        return
    end
    ltrl = 0.5*[-ltrh,-ltrw;ltrh,ltrw];
    ltrp = 0.5*[-ltrw,-ltrh;ltrw,ltrh];
    
    // Figure out the limits of the Scilab resolution (nx, ny)
    fig = scf(); clf();
    plot(1:10);
    fig.children(1).tight_limits = ["on","on","on"];
    fig.children(1).margins = [0,0,0,0];
    fig.children(1).grid = [0,0,0];
    fig.children(1).axes_visible = ["on","on","off"];
    fig.figure_position = [0,0];
    fig.children(1).isoview = "off";
    // Check what maximum axes we can get to reduce pixelation
    fig.axes_size = [3333,500];    // Try the X direction
    dt = 500;   // Sleep time to resize the axes... ugh
    sleep(dt);
    nx = fig.axes_size(1);
    fig.axes_size = [500,3333];    // Try the Y direction
    sleep(dt);
    ny = fig.axes_size(2);
    // We assume that landscape fits the screen best
    ar = ltrw/ltrh; // The aspect ratio that we must achieve (mm to pixels)
    // How do we choose the axes such that ny/nx == ar ?
    // Use the maxiumum #pixels in the Y direction that has the fewest pixels
    if ny < ar*nx then
        nx = round(ny/ar);
    else
        ny = round(ar*nx);
    end
    close(fig);
    
    function plotToSize(fig, nx, ny, dt, ltrl)
        fig.children(1).tight_limits = ["on","on","on"];
        fig.children(1).margins = [0,0,0,0];
        fig.children(1).grid = [0,0,0];
        fig.children(1).axes_visible = ["off","off","off"];
        fig.figure_position = [0,0];
        fig.children(1).isoview = "off";
        fig.axes_size = [nx, ny];     // Reduce pixelation
        sleep(dt);    // Setting the figure size takes a while
        fig.children(1).data_bounds = ltrl;
    endfunction

    // Min/max coordinates of the boards
    // Remember we have realigned the X coordinates to 0
    tb = findobj("tag","topBoard");
    xMin = min(tb.data.x);
    xMax = max(tb.data.x);
    dx = xMax - xMin;
    
    function [x,y] = printSector(c,ltrl,nx,ny,name)
        fig = scf(); clf();
        
        // If this is a Gee South sector, it is a single sector.
        // The North side always has 2 sectors (VNS or Gee).
        oneSector = %t;
        if c.typ == "VNS" | (min(c.s(:,2)) > 0) then
            oneSector = %f;
        end

        xyz = c.s - ones(c.s(:,1))*[c.w,c.e,-c.f];
        if c.typ == "VNS" then
            X = c.X;
            R = [X(2,1), -X(1,1), 0;
                 X(1,1),  X(2,1), 0;
                 0     ,  0     , 1];
            xyz = (R*xyz')';
        else
            R = [ cos(lambda), 0, sin(lambda);
                  0          , 1, 0;
                 -sin(lambda), 0, cos(lambda)];
            xyz = (R*xyz')';
        end
        if norm(xyz(:,1)-xyz(1,1)) > 1e-6 then
            error("Error in sector projection!");
            return;
        end
        x = xyz(:,2);
        x = x - (min(x) + max(x))/2;
        y = xyz(:,3);
        if oneSector then
            // Gee South sector, always 1 piece
            y = y - (min(y) + max(y))/2;
            plot(x, y);
        else
            // VNS or Gee North sectors, always 2 pieces
            y = y - max(y) + max(ltrl(:,2)) - 20;
            plot(x, y);
            y1 = -y;
            y1 = y1 - min(y1) + min(ltrl(:,2)) + 20;
            plot(-x, y1);
        end
        plotToSize(fig, nx, ny, dt, ltrl);
        xs2png(fig, name + ".png");
        close(fig);
    endfunction
    
    function [x,y] = geeBacking(objName,fn)
        gb = findobj("tag", objName);
        x = [];
        y = [];
        if gb == [] then
            return;
        end
        xyz = gb.data;
        R = [ cos(lambda), 0, sin(lambda);
              0          , 1, 0;
             -sin(lambda), 0, cos(lambda)];
        xyz = (R*xyz')';
        if norm(xyz(:,1)-xyz(1,1)) > 1e-6 then
            error("Error in sector projection!");
            return;
        end
        x = xyz(:,2);
        y = xyz(:,3);
        x = x - mean(x(1:2));
        y = y - min(y(4:5));
        fig = scf(); plot(x,y);
        plotToSize(fig, nx, ny, dt, ltrl);
        xs2png(fig,fn);
        close(fig);
    endfunction

    // Test page
    fig = scf(); clf();
    x = [-1;1;1;-1;-1];
    y = [-1;-1;1;1;-1];
    plot(50*x, 50*y, 'k');
    plotToSize(fig, nx, ny, dt, ltrl);
    xs2png(fig, "100x100TestPage.png");
    close(fig);

    if cn.typ == "VNS" then
        [x,y] = printSector(cn,ltrl,nx,ny,"northVnsSectors");
    else
        [x,y] = printSector(cn,ltrl,nx,ny,"northGeeSectors");
        [x1,y1] = geeBacking("geeBackingSouth","geeBackingSouth.png");
        if x1 <> [] then
            csvWrite([x1,y1],"geeBackingSouth.csv",",");
        end
    end
    csvWrite([x-mean(x),y-mean(y)],"northSector2d.csv",",");
    csvWrite(cn.s,"northSector3d.csv",",");
    if cs.typ == "VNS" then
        [x,y] = printSector(cs,ltrl,nx,ny,"southVnsSectors");
    else
        [x,y] = printSector(cs,ltrl,nx,ny,"southGeeSector");
        [x1,y1] = geeBacking("geeBackingNorthWest","geeBackingNorth.png");
        if x1 <> [] then
            csvWrite([x1,y1],"geeBackingNorth.csv",",");
        end
    end
    csvWrite([x-mean(x),y-mean(y)],"southSector2d.csv",",");
    csvWrite(cs.s,"southSector3d.csv",",");

    // 3D setup
    ax = findobj("tag", "axes");
    ax.tight_limits = ["on","on","on"];
    ax.margins = [0,0,0,0];
    ax.grid = [0,0,0];
    ax.axes_visible = ["off","off","off"];
    // We can't use 2d viewing because it would lead to portrait and 
    // we want landscape to avoid poor resolution
    fig3.figure_position = [0,0];
    ax.isoview = "off";
    ax.cube_scaling = "off";
    ax.rotation_angles = [0,270];   // North points right, as viewed from above
    fig3.axes_size = [nx,ny];
    sleep(dt);    // Setting the figure size takes a while

    // Center
    ctr = [0.5*dx,0];
    ax.data_bounds = [1;1]*ctr + ltrl;
    xs2png(fig3, "midCenter.png");

    // South side
    ctr = [0.1*dx,0];
    ax.data_bounds = [1;1]*ctr + ltrl;
    xs2png(fig3, "midSouth.png");

    // North side
    ctr = [0.9*dx,0];
    ax.data_bounds = [1;1]*ctr + ltrl;
    xs2png(fig3, "midNorth.png");

    ax.rotation_angles = [0,180];   // North points up, as viewed from above
    sleep(dt);

    // West side
    ctr = [0.95*xMax,-0.6*max(cn.s(:,2))];
    ax.data_bounds = [1;1]*ctr + ltrp;
    xs2png(fig3, "northWest.png");

    // East side
    ctr = [0.95*xMax,0.6*max(cn.s(:,2))];
    ax.data_bounds = [1;1]*ctr + ltrp;
    sleep(dt);    // Setting the figure size takes a while
    xs2png(fig3, "northEast.png");

    tb = findobj("tag", "topBoard");
    csvWrite(tb.data.x,"xTopBoard.csv",",");
    csvWrite(tb.data.y,"yTopBoard.csv",",");
    csvWrite(tb.data.z,"zTopBoard.csv",",");
    bb = findobj("tag", "bottomBoard");
    csvWrite(bb.data.x,"xBottomBoard.csv",",");
    csvWrite(bb.data.y,"yBottomBoard.csv",",");
    csvWrite(bb.data.z,"zBottomBoard.csv",",");

endfunction
