// Calibration of the VNS platform.
//
// Measure by putting the VNS platform on a table against the wall,
// assuming that they are perpendicular and vertical.  Detach the
// motor and put a tripod with a laser collimator on it.  Rotate
// it until you find the stationary spots on the wall and on the
// table.  Their connecting line is the cone axis.
//

wsp = 797;              // Wall stationary point (WSP) measured from the table
tsp = 1140;             // Table stationary point (TSP) measured from the wall
bd = 400;               // Distance between North bearings
bh = 65;                // Front bearing height
tspb = 868;             // Table stationary point to North bearing

// Positive Y axis is from TSP to the wall
// Positive Z axis is from the table to WSP
// Positive X axis is from TSP to the right when facing the wall
dx = 0;
dy = tsp;
dz = wsp;
Y1 = [dx; dy; dz];      // Vector from the table SP to the wall SP (cone axis)
printf("Cone axis angle = %6.4g\n", 180/%pi*atan(dz/dy));

psi = asin(bh/tspb);    // Angle of the line from TSP to bearing
chi = asin(0.5*bd/(tspb*cos(psi))); // Top view angle between TSP to bearing and wall
xb = 0.5*bd;
yb = tspb*cos(psi)*cos(chi);
zb = bh;
Y2 = [xb; yb; zb];      // Bearing location coordinates

[U,S,V] = svd(Y1);      // Unit vectors in plane perpendicular to cone axis are in U(:,2:3)
// Y2 + U(:,2:3)*theta12 = Y1*theta3
X = [-U(:,2:3), Y1];
theta = X\Y2;
p = Y2 - X(:,1:2)*theta(1:2);   // Center of Gee circle
r = norm(Y2 - p);       // Gee disc radius
printf("North Gee disc radius = %6.4g mm\n", r);

fig = scf(111); clf();
plot3d([-xb; 0; xb], [yb; 0; yb], [0;0;0], 'o-');
plot3d([-xb; 0; xb], [yb; 0; yb], [bh;0;bh], 'o-');
plot3d([0;0;0], [0;tsp;tsp], [0;0;wsp]);
plot3d([-xb;xb;p(1)], [yb;yb;p(2)], [zb; zb;p(3)]);
fig.children(1).isoview = "on";
